POSIX 风格的互斥服务。

---

### 详细描述

POSIX 风格的互斥服务。

互斥锁是一个 MUTual EXclusion 对象，用于保护共享数据结构免受并发修改，并实现临界区和监视器。

互斥锁有两种可能的状态：未锁定（未被任何任务拥有）和已锁定（被一个任务拥有）。互斥锁永远不能同时被两个不同的任务拥有。尝试锁定已被其他任务锁定的互斥锁的任务将被阻塞，直到后者先解锁互斥锁。

xkernel 互斥服务强制执行优先级继承协议以解决优先级反转问题。

---
### 函数文档

#### rt_mutex_acquire

> int rt_mutex_acquire (RT_MUTEX *mutex, RTIME timeout);

获取/锁定互斥锁（具有相对标量超时）。

此例程是 rt_mutex_acquire_timed() 的一个变体，接受以标量值表示的相对超时规范。

**参数**

- **mutex** 互斥锁描述符。
- **timeout** 以时钟刻度表示的延迟。传递 TM_INFINITE 会导致调用者无限期阻塞。传递 TM_NONBLOCK 会导致服务在互斥锁已被其他任务锁定的情况下立即返回，而不会阻塞。

**标签**

`xthread-only`, `switch-primary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_acquire_timed

> int rt_mutex_acquire_timed (RT_MUTEX *mutex, const struct timespec *abs_timeout);

获取/锁定互斥锁（具有绝对超时日期）。

尝试锁定互斥锁。调用任务被阻止，直到互斥锁可用，在这种情况下，它会在该服务返回之前再次被锁定。xkernel 互斥锁是隐式递归的，并实现优先级继承协议。

**参数**

- **mutex**	互斥锁描述符。
- **abs_timeout**   一个以秒/纳秒表示的绝对日期，基于 Alchemy 时钟，指定等待从堆中获取请求大小的块的时间限制。传递 NULL 会导致调用者无限期阻塞，直到有块可用。传递 { .tv_sec = 0, .tv_nsec = 0 } 会导致服务在块不可用的情况下立即返回，而不会阻塞。

**返回值**

成功时返回零。否则：
- 如果在互斥锁可用之前达到 abs_timeout，则返回 ETIMEDOUT。
- 如果超时为 { .tv_sec = 0, .tv_nsec = 0 } 并且互斥锁不是立即可用，则返回 EWOULDBLOCK。
- 如果为当前任务调用了 rt_task_unblock()，则返回 EINTR。
- 如果互斥锁不是有效的互斥锁描述符，则返回 EINVAL。
- 如果在调用者等待互斥锁时将其删除，则返回 EIDRM。在这种情况下，互斥锁在该服务返回时不再有效。
- 如果此服务应该阻塞但不是从一个 xkernel 线程调用，则返回 EPERM。

**标签**

`xthread-only`, `switch-primary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);

    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    ts.tv_sec += 2;  // 2 seconds
    ret = rt_mutex_acquire_timed(&mutex, &ts);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_acquire_until

> int rt_mutex_acquire_until (RT_MUTEX *mutex, RTIME timeout);

获取/锁定互斥锁（具有绝对标量超时）。

此例程是 rt_mutex_acquire_timed() 的变体，接受以标量值表示的绝对超时规范。

**参数**

- **mutex**	互斥锁描述符。
- **abs_timeout**   以时钟滴答表示的绝对日期。传递 TM_INFINITE 可使调用者无限期阻塞。传递 TM_NONBLOCK 可使服务立即返回而不阻塞，以防互斥锁已被其他任务锁定。

**标签**

`xthread-only`, `switch-primary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);

    RTIME timeout = rt_timer_read() + 2000000000;  // 2 seconds
    ret = rt_mutex_acquire_until(&mutex, timeout);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_bind

> int rt_mutex_bind (RT_MUTEX *mutex, const char *name, RTIME timeout);

绑定到一个互斥锁。

此例程创建一个新的描述符来引用由其符号名称标识的现有互斥锁。如果在入口处对象不存在，调用者可能会阻塞，直到创建了给定名称的互斥锁。

**参数**

- **mutex** 由操作填充的互斥锁描述符的地址。失败时，此内存的内容是未定义的。
- **name** 一个有效的以 NULL 结尾的名称，用于标识要绑定的互斥锁。此字符串应与传递给 rt_mutex_create() 的对象名称参数匹配。
- **timeout** 等待注册发生的时钟刻度数（见注）。传递 TM_INFINITE 会导致调用者无限期阻塞，直到对象被注册。传递 TM_NONBLOCK 会导致服务在对象在入口处未注册时立即返回，而不等待。

**返回值**

成功时返回零。否则：
- 如果在检索完成之前为当前任务调用了 rt_task_unblock()，则返回 -EINTR。
- 如果 timeout 等于 TM_NONBLOCK 并且在入口处未注册搜索的对象，则返回 -EWOULDBLOCK。
- 如果无法在指定的时间内检索到对象，则返回 -ETIMEDOUT。
- 如果此服务应阻塞，但未从 xkernel 线程调用，则返回 -EPERM。

**标签**

`xthread-nowait`, `switch-primary`

**注意**

超时值被解释为 Alchemy 时钟分辨率的倍数（参见 -alchemy-clock-resolution 选项，默认为 1 纳秒）。

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    RT_MUTEX mutex_task;

    ret = rt_mutex_bind(&mutex_task, "MyMutex", TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_bind %s\n", strerror(-ret));
        return;
    }

    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex_task, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex_task);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }

    rt_mutex_unbind(&mutex_task);
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_create

> int rt_mutex_create (RT_MUTEX *mutex, const char *name);

创建一个互斥锁。

创建一个互斥对象，允许多个任务同步访问共享资源。互斥锁在创建后处于未锁定状态。

**参数**

- **mutex** 互斥锁描述符的地址，成功创建后可用于唯一标识该对象。
- **name** 一个代表互斥锁符号名称的 ASCII 字符串。当非 NULL 和非空时，此字符串的副本用于将创建的互斥锁索引到对象注册表中。

**返回值**

成功时返回零。否则：
- 如果系统无法从主堆获取内存以创建互斥锁，则返回 -ENOMEM。
- 如果名称与已注册的互斥锁冲突，则返回 -EEXIST。
- 如果此服务是从无效的上下文调用的，例如中断或非 xkernel 线程，则返回 -EPERM。

**标签**

`xthread-only`, `mode-unrestricted`, `switch-secondary`

**注意**

属于同一 xkernel 会话的多个进程可以共享互斥锁。

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_delete

> int rt_mutex_delete (RT_MUTEX *mutex);

删除一个互斥锁。

此例程删除由 rt_mutex_create() 调用先前创建的互斥对象。

**参数**

- **mutex** 互斥锁描述符。

**返回值**

成功时返回零。否则：
- 如果 mutex 不是有效的互斥锁描述符，则返回 -EINVAL。
- 如果此服务是从异步上下文调用的，则返回 -EPERM。
- 如果尝试销毁由 mutex 引用的对象时，该对象仍被引用（例如，另一个任务正在使用 rt_mutex_acquire()、rt_mutex_acquire_timed() 或 rt_mutex_acquire_until()），则返回 -EBUSY。

**标签**

`mode-unrestricted`, `switch-secondary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_inquire

> int rt_mutex_inquire (RT_MUTEX *mutex, RT_MUTEX_INFO *info);

查询互斥锁状态。

此例程返回指定互斥锁的状态信息。

**参数**

- **mutex** 互斥锁描述符。
- **info** 一个指向返回缓冲区的指针，用于复制信息。

**返回值**

成功时返回零，并将状态信息写入 info 指向的结构体。否则：
- 如果 mutex 不是有效的互斥锁描述符，则返回 -EINVAL。
- 如果此服务是从中断上下文调用的，则返回 -EPERM。

**标签**

`xthread-only`, `switch-primary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    RT_MUTEX_INFO info;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);
    

    ret = rt_mutex_inquire(&mutex, &info);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_inquire %s\n", strerror(-ret));
        return;
    }
    printf("Task %d: Get mutex info\n",task_id);
    printf("Task %d: Name :%s, Owner:%p\n",task_id,info.name,info.owner);

    ret = rt_mutex_acquire(&mutex, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_release

> int rt_mutex_release (RT_MUTEX *mutex);

释放/解锁互斥锁。

此例程释放先前通过调用 rt_mutex_acquire() 或 rt_mutex_acquire_until() 锁定的互斥锁对象。如果互斥锁处于挂起状态，则按优先级顺序立即解除第一个等待任务的阻塞，并将互斥锁的所有权转移给该任务；否则，互斥锁将处于未锁定状态。

**参数**

- **mutex**: 互斥锁描述符。

**返回值**

成功时返回零。否则：
- 如果 mutex 不是有效的互斥锁描述符，则返回 -EINVAL。
- 如果 mutex 不属于当前任务，或者更一般地说，如果此服务是在无法拥有任何互斥锁的上下文（例如中断上下文）中调用的，则返回 -EPERM。

**标签**

`xthread-only`, `switch-primary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    
    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---

#### rt_mutex_unbind

> int rt_mutex_unbind (RT_MUTEX *mutex);

解除互斥锁绑定。

**参数**

- **mutex**: 互斥锁描述符。

此例程释放先前对互斥锁的绑定。此调用返回后，描述符不再有效，无法用于引用该对象。

**标签**

`mode-unrestricted`, `switch-secondary`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/mutex.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0

RT_MUTEX mutex;
int shared_resource = 0;

void task_function(void *arg)
{
    int task_id = (int)(long)arg;
    int ret;
    RT_MUTEX mutex_task;

    ret = rt_mutex_bind(&mutex_task, "MyMutex", TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_bind %s\n", strerror(-ret));
        return;
    }

    printf("Task %d: Try to acquire mutex...\n", task_id);
    
    ret = rt_mutex_acquire(&mutex_task, TM_INFINITE);
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_acquire %s\n", strerror(-ret));
        return;
    }
    
    printf("Task %d: Successfully acquired the mutex\n", task_id);
    
    // 访问共享资源
    shared_resource++;
    printf("Task %d: Increase the shared resource value to %d\n", task_id, shared_resource);
    
    // 模拟工作
    rt_task_sleep(1000000000);  // 睡眠1秒
    
    printf("Task %d: Release the mutex\n", task_id);
    ret = rt_mutex_release(&mutex_task);
    if (ret) {
       fprintf(stderr, "Task: rt_mutex_release %s\n", strerror(-ret));
    }

    rt_mutex_unbind(&mutex_task);
}

int main(int argc, char *argv[])
{
    RT_TASK task1, task2;
    int ret;

    // 创建互斥锁
    ret = rt_mutex_create(&mutex, "MyMutex");
    if (ret) {
        fprintf(stderr, "Task: rt_mutex_create %s\n", strerror(-ret));
        return EXIT_FAILURE;
    }

    // 创建任务1
    ret = rt_task_create(&task1, "Task1", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        return EXIT_FAILURE;
    }

    // 创建任务2
    ret = rt_task_create(&task2, "Task2", TASK_STKSZ, TASK_PRIO, TASK_MODE);
    if (ret) {
        fprintf(stderr, "Task: rt_task_create %s\n", strerror(-ret));
        rt_mutex_delete(&mutex);
        rt_task_delete(&task1);
        return EXIT_FAILURE;
    }

    // 启动任务
    rt_task_start(&task1, &task_function, (void *)1);
    rt_task_start(&task2, &task_function, (void *)2);

    // 等待任务完成
    rt_task_join(&task1);
    rt_task_join(&task2);

    // 清理
    rt_mutex_delete(&mutex);
    rt_task_delete(&task1);
    rt_task_delete(&task2);

    printf("Final shared resource value: %d\n", shared_resource);

    return EXIT_SUCCESS;
}
```

---